#ifndef CALLBACK_BRIDGE_H
#define CALLBACK_BRIDGE_H

#include <vector>
#include <nan.h>
#include <condition_variable>
#include <algorithm>

#define COMMA ,

template <typename T, typename L = void*>
class CallbackBridge {
  public:
    CallbackBridge(Nan::Callback*, bool);
    virtual ~CallbackBridge();

    // Executes the callback
    T operator()(std::vector<void*>);

  protected:
    // We will expose a bridge object to the JS callback that wraps this instance so we don't loose context.
    // This is the V8 constructor for such objects.
    static Nan::MaybeLocal<v8::Function> get_wrapper_constructor();
    static void async_gone(uv_handle_t *handle);
    static NAN_METHOD(New);
    static NAN_METHOD(ReturnCallback);
    static Nan::Persistent<v8::Function> wrapper_constructor;
    Nan::Persistent<v8::Object> wrapper;

    // The callback that will get called in the main thread after the worker thread used for the sass
    // compilation step makes a call to uv_async_send()
    static void dispatched_async_uv_callback(uv_async_t*);

    // The V8 values sent to our ReturnCallback must be read on the main thread not the sass worker thread.
    // This gives a chance to specialized subclasses to transform those values into whatever makes sense to
    // sass before we resume the worker thread.
    virtual T post_process_return_value(v8::Local<v8::Value>) const =0;


    virtual std::vector<v8::Local<v8::Value>> pre_process_args(std::vector<L>) const =0;

    Nan::Callback* callback;
    bool is_sync;

    std::mutex cv_mutex;
    std::condition_variable condition_variable;
    uv_async_t *async;
    std::vector<L> argv;
    bool has_returned;
    T return_value;
};

template <typename T, typename L>
Nan::Persistent<v8::Function> CallbackBridge<T, L>::wrapper_constructor;

template <typename T, typename L>
CallbackBridge<T, L>::CallbackBridge(Nan::Callback* callback, bool is_sync) : callback(callback), is_sync(is_sync) {
  // This assumes the main thread will be the one instantiating the bridge
  if (!is_sync) {
    this->async = new uv_async_t;
    this->async->data = (void*) this;
    uv_async_init(uv_default_loop(), this->async, (uv_async_cb) dispatched_async_uv_callback);
  }

  v8::Local<v8::Function> func = CallbackBridge<T, L>::get_wrapper_constructor().ToLocalChecked();
  wrapper.Reset(Nan::NewInstance(func).ToLocalChecked());
  Nan::SetInternalFieldPointer(Nan::New(wrapper), 0, this);
}

template <typename T, typename L>
CallbackBridge<T, L>::~CallbackBridge() {
  delete this->callback;
  this->wrapper.Reset();

  if (!is_sync) {
    uv_close((uv_handle_t*)this->async, &async_gone);
  }
}

template <typename T, typename L>
T CallbackBridge<T, L>::operator()(std::vector<void*> argv) {
  // argv.push_back(wrapper);
  if (this->is_sync) {
    Nan::EscapableHandleScope scope;
    std::vector<v8::Local<v8::Value>> argv_v8 = pre_process_args(argv);
    argv_v8.push_back(Nan::New(wrapper));

    return this->post_process_return_value(
      scope.Escape(this->callback->Call(argv_v8.size(), &argv_v8[0]))
    );
  }

  this->argv = argv;

  std::unique_lock<std::mutex> lock(this->cv_mutex);
  this->has_returned = false;
  uv_async_send(this->async);
  this->condition_variable.wait(lock, [this] { return this->has_returned; });

  return this->return_value;
}

template <typename T, typename L>
void CallbackBridge<T, L>::dispatched_async_uv_callback(uv_async_t *req) {
  CallbackBridge* bridge = static_cast<CallbackBridge*>(req->data);

  Nan::HandleScope scope;
  Nan::TryCatch try_catch;

  std::vector<v8::Local<v8::Value>> argv_v8 = bridge->pre_process_args(bridge->argv);
  argv_v8.push_back(Nan::New(bridge->wrapper));

  bridge->callback->Call(argv_v8.size(), &argv_v8[0]);

  if (try_catch.HasCaught()) {
    Nan::FatalException(try_catch);
  }
}

template <typename T, typename L>
NAN_METHOD(CallbackBridge<T COMMA L>::ReturnCallback) {

  CallbackBridge<T, L>* bridge = static_cast<CallbackBridge<T, L>*>(Nan::GetInternalFieldPointer(info.This(), 0));
  Nan::TryCatch try_catch;

  bridge->return_value = bridge->post_process_return_value(info[0]);

  {
    std::lock_guard<std::mutex> lock(bridge->cv_mutex);
    bridge->has_returned = true;
  }

  bridge->condition_variable.notify_all();

  if (try_catch.HasCaught()) {
    Nan::FatalException(try_catch);
  }
}

template <typename T, typename L>
Nan::MaybeLocal<v8::Function> CallbackBridge<T, L>::get_wrapper_constructor() {
  if (wrapper_constructor.IsEmpty()) {
    v8::Local<v8::FunctionTemplate> tpl = Nan::New<v8::FunctionTemplate>(New);
    tpl->SetClassName(Nan::New("CallbackBridge").ToLocalChecked());
    tpl->InstanceTemplate()->SetInternalFieldCount(1);

    Nan::SetPrototypeTemplate(tpl, "success",
      Nan::GetFunction(Nan::New<v8::FunctionTemplate>(ReturnCallback)).ToLocalChecked()
    );

    wrapper_constructor.Reset(Nan::GetFunction(tpl).ToLocalChecked());
  }

  return Nan::New(wrapper_constructor);
}

template <typename T, typename L>
NAN_METHOD(CallbackBridge<T COMMA L>::New) {
  info.GetReturnValue().Set(info.This());
}

template <typename T, typename L>
void CallbackBridge<T, L>::async_gone(uv_handle_t *handle) {
  delete (uv_async_t *)handle;
}

#endif
