#include <nan.h>
#include "number.h"
#include "../create_string.h"

namespace SassTypes
{
  Number::Number(Sass_Value* v) : SassValueWrapper(v) {}

  Sass_Value* Number::construct(const std::vector<v8::Local<v8::Value>> raw_val) {
    double value = 0;
    char const* unit = "";

    if (raw_val.size() >= 1) {
      if (!raw_val[0]->IsNumber()) {
        throw std::invalid_argument("First argument should be a number.");
      }

      value = Nan::To<double>(raw_val[0]).FromJust();

      if (raw_val.size() >= 2) {
        if (!raw_val[1]->IsString()) {
          throw std::invalid_argument("Second argument should be a string.");
        }

        unit = create_string(raw_val[1]);
      }
    }

    return sass_make_number(value, unit);
  }

  void Number::initPrototype(v8::Local<v8::FunctionTemplate> proto) {
    Nan::SetPrototypeMethod(proto, "getValue", GetValue);
    Nan::SetPrototypeMethod(proto, "getUnit", GetUnit);
    Nan::SetPrototypeMethod(proto, "setValue", SetValue);
    Nan::SetPrototypeMethod(proto, "setUnit", SetUnit);
  }

  NAN_METHOD(Number::GetValue) {
    info.GetReturnValue().Set(Nan::New<v8::Number>(sass_number_get_value(unwrap(info.This())->value)));
  }

  NAN_METHOD(Number::GetUnit) {
    info.GetReturnValue().Set(Nan::New<v8::String>(sass_number_get_unit(unwrap(info.This())->value)).ToLocalChecked());
  }

  NAN_METHOD(Number::SetValue) {

    if (info.Length() != 1) {
      return Nan::ThrowError(Nan::New("Expected just one argument").ToLocalChecked());
    }

    if (!info[0]->IsNumber()) {
      return Nan::ThrowError(Nan::New("Supplied value should be a number").ToLocalChecked());
    }

    sass_number_set_value(unwrap(info.This())->value, Nan::To<double>(info[0]).FromJust());
  }

  NAN_METHOD(Number::SetUnit) {
    if (info.Length() != 1) {
      return Nan::ThrowError(Nan::New("Expected just one argument").ToLocalChecked());
    }

    if (!info[0]->IsString()) {
      return Nan::ThrowError(Nan::New("Supplied value should be a string").ToLocalChecked());
    }

    sass_number_set_unit(unwrap(info.This())->value, create_string(info[0]));
  }
}
