/* solium-disable security/no-block-members */

pragma solidity ^0.4.24;

import "../token/ERC20/SafeERC20.sol";
import "../ownership/Ownable.sol";
import "../math/SafeMath.sol";


/**
 * @title TokenVesting
 * @dev A token holder contract that can release its token balance gradually like a
 * typical vesting scheme, with a cliff and vesting period. Optionally revocable by the
 * owner.
 */
contract TokenVesting is Ownable {
  using SafeMath for uint256;
  using SafeERC20 for IERC20;

  event Released(uint256 amount);
  event Revoked();

  // beneficiary of tokens after they are released
  address private beneficiary_;

  uint256 private cliff_;
  uint256 private start_;
  uint256 private duration_;

  bool private revocable_;

  mapping (address => uint256) private released_;
  mapping (address => bool) private revoked_;

  /**
   * @dev Creates a vesting contract that vests its balance of any ERC20 token to the
   * _beneficiary, gradually in a linear fashion until _start + _duration. By then all
   * of the balance will have vested.
   * @param _beneficiary address of the beneficiary to whom vested tokens are transferred
   * @param _cliffDuration duration in seconds of the cliff in which tokens will begin to vest
   * @param _start the time (as Unix time) at which point vesting starts
   * @param _duration duration in seconds of the period in which the tokens will vest
   * @param _revocable whether the vesting is revocable or not
   */
  constructor(
    address _beneficiary,
    uint256 _start,
    uint256 _cliffDuration,
    uint256 _duration,
    bool _revocable
  )
    public
  {
    require(_beneficiary != address(0));
    require(_cliffDuration <= _duration);

    beneficiary_ = _beneficiary;
    revocable_ = _revocable;
    duration_ = _duration;
    cliff_ = _start.add(_cliffDuration);
    start_ = _start;
  }

  /**
   * @return the beneficiary of the tokens.
   */
  function beneficiary() public view returns(address) {
    return beneficiary_;
  }

  /**
   * @return the cliff time of the token vesting.
   */
  function cliff() public view returns(uint256) {
    return cliff_;
  }

  /**
   * @return the start time of the token vesting.
   */
  function start() public view returns(uint256) {
    return start_;
  }

  /**
   * @return the duration of the token vesting.
   */
  function duration() public view returns(uint256) {
    return duration_;
  }

  /**
   * @return true if the vesting is revocable.
   */
  function revocable() public view returns(bool) {
    return revocable_;
  }

  /**
   * @return the amount of the token released.
   */
  function released(address _token) public view returns(uint256) {
    return released_[_token];
  }

  /**
   * @return true if the token is revoked.
   */
  function revoked(address _token) public view returns(bool) {
    return revoked_[_token];
  }

  /**
   * @notice Transfers vested tokens to beneficiary.
   * @param _token ERC20 token which is being vested
   */
  function release(IERC20 _token) public {
    uint256 unreleased = releasableAmount(_token);

    require(unreleased > 0);

    released_[_token] = released_[_token].add(unreleased);

    _token.safeTransfer(beneficiary_, unreleased);

    emit Released(unreleased);
  }

  /**
   * @notice Allows the owner to revoke the vesting. Tokens already vested
   * remain in the contract, the rest are returned to the owner.
   * @param _token ERC20 token which is being vested
   */
  function revoke(IERC20 _token) public onlyOwner {
    require(revocable_);
    require(!revoked_[_token]);

    uint256 balance = _token.balanceOf(address(this));

    uint256 unreleased = releasableAmount(_token);
    uint256 refund = balance.sub(unreleased);

    revoked_[_token] = true;

    _token.safeTransfer(owner(), refund);

    emit Revoked();
  }

  /**
   * @dev Calculates the amount that has already vested but hasn't been released yet.
   * @param _token ERC20 token which is being vested
   */
  function releasableAmount(IERC20 _token) public view returns (uint256) {
    return vestedAmount(_token).sub(released_[_token]);
  }

  /**
   * @dev Calculates the amount that has already vested.
   * @param _token ERC20 token which is being vested
   */
  function vestedAmount(IERC20 _token) public view returns (uint256) {
    uint256 currentBalance = _token.balanceOf(this);
    uint256 totalBalance = currentBalance.add(released_[_token]);

    if (block.timestamp < cliff_) {
      return 0;
    } else if (block.timestamp >= start_.add(duration_) || revoked_[_token]) {
      return totalBalance;
    } else {
      return totalBalance.mul(block.timestamp.sub(start_)).div(duration_);
    }
  }
}
